function [mtlenm,time,time_ringclose,mtstate_ringclose,time_visibleMTnucl] = simgTuRCnucleationdynamics(TbConc,kon,GLong_TB,GLat_TB,GLong_gTB,kringclosure,GgTuRCClosure,simTime)

%% Simulation of gTuRC dynamics
% Current version (v1.1)
% Akanksha Thawani, April 2020
%
% The following code simulates microtubule nucleation dynamics from gTuRC
%
% version 1.1 (April 28, 2020): written by Akanksha Thawani
% microtubule dynamics is based on simple code written by
% David Odde et al, ref: VanBuren et al., PNAS 2002
%
%% Assumptions and parameters
% Current version (v1.1): simulation for nucleation kinetics from gTuRC
% Features to include:
% 1. Open to closed transition, here it matters what
% geometry to include - no lateral contacts prior to closure, or have some
% lateral contacts prior to closure
% 2. Bond energy between gamma- ab tubulin: change this factor
% 3. Polymerization dynamics: kon, GLong_TB, GLat_TB
% these will have to be optimized to match my experimental observations for
% growth speed
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Set output values to NaN
time_ringclose = NaN;
mtstate_ringclose=NaN;

%% Set standard parameter values %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

Npf = 13; % number of protofilaments
pitch = 3; % helical pitch of the microtubule, in ab-tubulin dimers per turn
d0 = 4; % nm, length of a single alpha-beta tubulin subunit
GTP = 2; % value assigned to tubulin monomers bound with GTP
GDP = 1; % value assigned to tubulin monomers bound with GDP
riseperpf = (pitch*d0)/(Npf); % nm, length increase due to the helical pitch of the MT, per protofilament

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Set gTuRC specific parameters %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

seedlayergTB = 1;
OPENgTuRC = 3;
CLOSEDgTuRC = 4;

OpenTemplateLeftNeighbor = [0 0 0 0 0 0 0 0 0 0 0 0 0];
OpenTemplateRightNeighbor = [0 0 0 0 0 0 0 0 0 0 0 0 0];

closedflag = 0;

initseg = 2000; % nm of MT length
MTlayers = round(initseg/d0);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Initialize random number generator%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
try task = getCurrentTask;id = task.ID;catch;id = 1;end
cc = clock;kk = sum([cc(2),cc(3),cc(4),cc(5)])*2^(cc(6)/2)*id^3;
while kk>2^32;kk = kk/100;end
rng(kk,'twister')

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Initialize arrays and counting variables%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

totSample = min([round(simTime/0.0001),1e6]);

mtlenm = zeros(totSample,1);
time = zeros(totSample,1);
added = zeros(totSample,1);
lost = zeros(totSample,1);
closure = zeros(totSample,1);

numadded = 0;
numlost = 0;
timetrack = 0;
tottime = 0;
sampleIndex = 1;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Initialize gTuRC and MT arrays %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[MT.gTuRC,MT.MT,MT.gTuRCMT,MT.d,MT.mtalpha,MT.mtbeta,MT.mtgamma] = initializegTuRCMT(seedlayergTB,MTlayers,d0,Npf,pitch,OPENgTuRC);

gturc = MT.gTuRC;
mtgturc = MT.gTuRCMT;
mtgamma = MT.mtgamma;
d = MT.d;

mt = MT.MT;
mtalpha = MT.mtalpha;
mtbeta = MT.mtbeta;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
while tottime<simTime
        
    mtgturcMask = (mtgturc>0);
    
    [Etot,~,~,latbondMask,Eringclosure] = mtEnergy(Npf,pitch,mtgturc,mtgturcMask,gturc,mt,...
                        mtalpha,mtbeta,mtgamma,closedflag,...
                        OpenTemplateLeftNeighbor,OpenTemplateRightNeighbor,...
                        GLong_TB,GLat_TB,GLong_gTB,GgTuRCClosure);
                    
    
    %% Tubulin Association Events %%%%%%%%%%%%%%%%
    kplus = kon*TbConc*ones(1,Npf);
    Ra = 1-rand(size(kplus));
    
    timeToBind = -log(Ra)./kplus;
    [timeA,pfA] = min(timeToBind);
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% Tubulin Dissociation Events %%%%%%%%%%%%
    K = exp(-Etot);
    K(mtalpha==1 | mtgturc==0) = NaN;
    
    kminus = kon./K;
    
    Rd = 1-rand(size(K));
    
    timeToUnbind = -log(Rd)./kminus;
    [timeD,levelD] = min(timeToUnbind);
    [timeD,pfD] = min(timeD);
    levelD = levelD(pfD);
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% Ring Closure Event %%%%%%%%%%%%
    if ~closedflag
        Krc = exp(-Eringclosure);

        kclose = kringclosure.*Krc;
        Rrc = 1-rand(size(Krc));
        timeToRingClose = -log(Rrc)./kclose;
    else
        timeToRingClose = NaN;
    end

    %% Execute Event %%%%%%%%%%%%%%%%%%%%
    eventTime = min([timeA,timeD,timeToRingClose]);

    switch eventTime
        case timeA %add a subunit
            pfEnds = sum(mtgturcMask);
            levelA = pfEnds(pfA)+1;
            mtgturc(levelA:levelA+1,pfA) = GTP;
            
            mtalpha(levelA,pfA) = 1;
            mtbeta(levelA+1,pfA) = 1;
            d(levelA:levelA+1,pfA) = d0;
            
            numadded = numadded+1;
            timetrack = timetrack+eventTime;
        case timeD %remove a subunit, and everything above (distal to) it
            mtgturc(levelD-1:end,pfD) = 0;
            mtalpha(levelD-1:end,pfD) = 0;
            mtbeta(levelD-1:end,pfD) = 0;
            d(levelD-1:end,pfD) = 0;
            
            numlost = numlost+1;
            timetrack = timetrack+eventTime;
        case timeToRingClose %gTuRC ring closes
            closedflag = 1;

            timetrack = timetrack+eventTime;
            % output the time; output the state of gTuRC
            pfEnds = sum(mtgturcMask); [tallpfidx,~] = max(pfEnds);
            mtstate_ringclose = mtgturc(1:tallpfidx,:);
            time_ringclose = tottime+timetrack;
            
            mtgturc(1,:) = CLOSEDgTuRC;
    end
    
    mt = mtgturc(seedlayergTB+1:end,:);
    gturc = mtgturc(seedlayergTB,:);
    
    %% Sample Simulation Progress %%%%%%%%%%%%%
    
    tottime = tottime+timetrack;
    sampleIndex = sampleIndex+1;

    pflenm = sum(d)+fliplr(0:Npf-1)*riseperpf;

    mtlenm(sampleIndex) = mean(pflenm);

    time(sampleIndex) = tottime;
    added(sampleIndex) = numadded;
    lost(sampleIndex) = numlost;
    closure(sampleIndex) = closedflag;

    timetrack = 0;
    numadded = 0;
    numlost = 0;
        
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% Determining if microtubule has assembled to max size %%%%%%%%%%%%%%%%%
    mtgturcMask = (mtgturc>0);
    [rmax,~] = size(mtgturcMask);
    pfEnds = sum(mtgturcMask);
    pfmax = max(pfEnds);
    if pfmax >= rmax
        warning('Microtubule assembled to maximum length')
        break
    end
    
    if sampleIndex==numel(mtlenm)
        % if array size is exceeded, then pad arrays to avoid significant loss
        % of simulation speed
        mtlenm = [mtlenm;zeros(100000,1)];
        time = [time;zeros(100000,1)];
        added = [added;zeros(100000,1)];
        lost = [lost;zeros(100000,1)];
        closure = [closure;zeros(100000,1)];
        
    end
    
end


% clear out any padded zeros at the end of the output arrays
    mtlenm = mtlenm(1:sampleIndex);
    time = time(1:sampleIndex);
    added = added(1:sampleIndex);
    lost = lost(1:sampleIndex);
    closure = closure(1:sampleIndex);

    subplot(3,1,1)
    plot(time,mtlenm/1000,'b-')
    xlabel('Time (sec)','FontSize',14)
    ylabel('MT Length (\mum)','FontSize',14)
    set(gca,'XLim',[0 ceil(max(time))])

    time_visibleMTnucl=(time(find(mtlenm>=200,1)));
    if isempty(time_visibleMTnucl)
        time_visibleMTnucl = NaN;
    end
        
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% initializegTuRC+MTseg.m %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [gTuRC,MT,gTuRCMT,d,mtalpha,mtbeta,mtgamma] = initializegTuRCMT(seedlayergTB,MTlayers,d0,Npf,pitch,OPENgTuRC)

gTuRCMT = zeros(seedlayergTB+MTlayers+pitch,Npf);

d = gTuRCMT;

gTuRCMT(1:seedlayergTB,:) = OPENgTuRC;
gTuRC = gTuRCMT(1:seedlayergTB,:);

MT = gTuRCMT(seedlayergTB+1:end,:);
d(1:seedlayergTB,:) = d0;
mtgamma = gTuRCMT>0;
mtalpha = zeros(size(gTuRCMT));
mtbeta = zeros(size(gTuRCMT));

end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% mtEnergy.m %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [Etot,Elong,Elat,latbondMask,Eringclosure] = mtEnergy(Npf,pitch,mtgturc,mtgturcMask,gturc,mt,mtalpha,mtbeta,mtgamma,closedflag,OpenTemplateLeftNeighbor,OpenTemplateRightNeighbor,GLong_TB,GLat_TB,GLong_gTB,GgTuRCClosure)

    
if ~closedflag
    bondMask1 = [mtgturcMask(:,2:Npf),mtgturcMask(:,1)]; %"looking" to the right
    bondMask1(1+pitch:end,Npf) = bondMask1(1:end-pitch,Npf); bondMask1(1:pitch,Npf) = 0;
    bondMask1closed  = bondMask1.*mtgturcMask;
    bondMask1 = bondMask1closed.*repmat(OpenTemplateRightNeighbor,[size(mtgturcMask,1),1]);

    bondMask2 = [mtgturcMask(:,Npf),mtgturcMask(:,1:Npf-1)]; %"looking" to the left
    bondMask2(1:end-pitch,1) = bondMask2(1+pitch:end,1);
    bondMask2closed = bondMask2.*mtgturcMask;    
    bondMask2 = bondMask2closed.*repmat(OpenTemplateLeftNeighbor,[size(mtgturcMask,1),1]);
    
    latbondMask = bondMask1+bondMask2;
    
    
    
else
    bondMask1 = [mtgturcMask(:,2:Npf),mtgturcMask(:,1)]; %"looking" to the right
    bondMask1(1+pitch:end,Npf) = bondMask1(1:end-pitch,Npf); bondMask1(1:pitch,Npf) = 0;
    bondMask1 = bondMask1.*mtgturcMask;

    bondMask2 = [mtgturcMask(:,Npf),mtgturcMask(:,1:Npf-1)]; %"looking" to the left
    bondMask2(1:end-pitch,1) = bondMask2(1+pitch:end,1);
    bondMask2 = bondMask2.*mtgturcMask;
    latbondMask = bondMask1+bondMask2;
    
    
end



Elat1 = bondMask1*(GLat_TB/2);
Elat2 = bondMask2*(GLat_TB/2);

Elat = Elat1+Elat2;
Elat(3:2:end,:) = (Elat(2:2:end-1,:)+Elat(3:2:end,:)); %assign energy to beta-tubulin
Elat = (Elat.*mtbeta);

Elong = mtbeta*GLong_TB;
Elong(3,:) = mtbeta(3,:)*GLong_gTB;

Elong = Elong.*mtgturcMask;

%% calculate energy for dissociation for each dimer and assign to beta tubulin
E1 = Elat;
E1 = flipud(E1);
E1Sum = cumsum(E1);
E1Sum = flipud(E1Sum).*mtbeta;

Etot = E1Sum+Elong;

Etot(1,:) = NaN;

%% calculate energy for ring closure
if ~closedflag
    Elat_ringclosure = ((bondMask1closed-bondMask1))*(GLat_TB/2);

    Elat_ringclosure(1,:) = 0;
    Elat_ringclosure(3:2:end,:) = (Elat_ringclosure(2:2:end-1,:)+Elat_ringclosure(3:2:end,:)); %assign energy to beta-tubulin
    Elat_ringclosure = (Elat_ringclosure.*mtbeta);

    Eringclosure = GgTuRCClosure+sum(Elat_ringclosure(:));
else
    Eringclosure = 0;
end

if any(Etot(:)>0)
    Etot(Etot>0) = 0;
    warning('Positive subunit free energy value(s) occurred, consider using different input values for GLong, GLat')
end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

